const express = require('express');
const multer = require('multer');
const path = require('path');
const fs = require('fs');
const router = express.Router();
const Assignment = require('../../models/teacher/Assignment');
const AssignmentSubmission = require('../../models/student/AssignmentSubmission');
const Student = require('../../models/student/Student');
const { teacherAuth } = require('../../middleware/auth');

// Configure multer for file uploads
const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    const uploadDir = path.join(__dirname, '../../uploads/assignments/teacher');
    if (!fs.existsSync(uploadDir)) {
      fs.mkdirSync(uploadDir, { recursive: true });
    }
    cb(null, uploadDir);
  },
  filename: (req, file, cb) => {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    cb(null, 'assignment-' + uniqueSuffix + path.extname(file.originalname));
  }
});

const upload = multer({
  storage: storage,
  limits: {
    fileSize: 10 * 1024 * 1024, // 10MB limit
  },
  fileFilter: (req, file, cb) => {
    const allowedTypes = [
      'application/pdf',
      'application/msword',
      'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
      'application/vnd.ms-powerpoint',
      'application/vnd.openxmlformats-officedocument.presentationml.presentation',
      'text/plain',
      'image/jpeg',
      'image/png',
      'image/gif'
    ];
    
    if (allowedTypes.includes(file.mimetype)) {
      cb(null, true);
    } else {
      cb(new Error('Invalid file type. Only PDF, Word, PowerPoint, text, and image files are allowed.'));
    }
  }
});

// Create new assignment
router.post('/', teacherAuth, upload.array('attachments', 5), async (req, res) => {
  try {
    const { title, description, subject, teacherId, assignedStudents, dueDate, maxMarks, instructions, allowLateSubmission, lateSubmissionPenalty } = req.body;
    
    const parsedStudents = JSON.parse(assignedStudents || '[]');
    
    const attachments = req.files ? req.files.map(file => ({
      filename: file.filename,
      originalName: file.originalname,
      path: file.path,
      size: file.size,
      mimetype: file.mimetype
    })) : [];
    
    const assignment = new Assignment({
      title,
      description,
      subject,
      teacherId,
      assignedStudents: parsedStudents,
      dueDate: new Date(dueDate),
      maxMarks: parseInt(maxMarks) || 100,
      attachments,
      instructions,
      allowLateSubmission: allowLateSubmission === 'true',
      lateSubmissionPenalty: parseInt(lateSubmissionPenalty) || 0
    });
    
    await assignment.save();
    res.status(201).json({ success: true, assignment });
  } catch (error) {
    res.status(400).json({ success: false, error: error.message });
  }
});

// Get assignments for a teacher
router.get('/', teacherAuth, async (req, res) => {
  try {
    const { teacherId, status, page = 1, limit = 10 } = req.query;
    const filter = {};
    
    if (teacherId) filter.teacherId = teacherId;
    if (status) filter.status = status;
    
    const skip = (page - 1) * limit;
    
    const assignments = await Assignment.find(filter)
      .sort({ createdAt: -1 })
      .skip(skip)
      .limit(parseInt(limit));
    
    // Get submission stats for each assignment
    const assignmentsWithStats = await Promise.all(
      assignments.map(async (assignment) => {
        const stats = await assignment.getSubmissionStats();
        return {
          ...assignment.toObject(),
          submissionStats: stats
        };
      })
    );
    
    const total = await Assignment.countDocuments(filter);
    
    res.json({
      success: true,
      assignments: assignmentsWithStats,
      pagination: {
        page: parseInt(page),
        limit: parseInt(limit),
        total,
        pages: Math.ceil(total / limit)
      }
    });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
});

// Get single assignment by ID
router.get('/:id', async (req, res) => {
  try {
    const assignment = await Assignment.findById(req.params.id);
    if (!assignment) {
      return res.status(404).json({ success: false, error: 'Assignment not found' });
    }
    
    const stats = await assignment.getSubmissionStats();
    
    res.json({
      success: true,
      assignment: {
        ...assignment.toObject(),
        submissionStats: stats
      }
    });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
});

// Update assignment
router.put('/:id', upload.array('attachments', 5), async (req, res) => {
  try {
    const assignment = await Assignment.findById(req.params.id);
    if (!assignment) {
      return res.status(404).json({ success: false, error: 'Assignment not found' });
    }
    
    const { title, description, subject, assignedStudents, dueDate, maxMarks, instructions, allowLateSubmission, lateSubmissionPenalty } = req.body;
    
    // Update basic fields
    if (title) assignment.title = title;
    if (description) assignment.description = description;
    if (subject) assignment.subject = subject;
    if (assignedStudents) assignment.assignedStudents = JSON.parse(assignedStudents);
    if (dueDate) assignment.dueDate = new Date(dueDate);
    if (maxMarks) assignment.maxMarks = parseInt(maxMarks);
    if (instructions) assignment.instructions = instructions;
    if (allowLateSubmission !== undefined) assignment.allowLateSubmission = allowLateSubmission === 'true';
    if (lateSubmissionPenalty !== undefined) assignment.lateSubmissionPenalty = parseInt(lateSubmissionPenalty);
    
    // Handle new file uploads
    if (req.files && req.files.length > 0) {
      const newAttachments = req.files.map(file => ({
        filename: file.filename,
        originalName: file.originalname,
        path: file.path,
        size: file.size,
        mimetype: file.mimetype
      }));
      assignment.attachments.push(...newAttachments);
    }
    
    await assignment.save();
    res.json({ success: true, assignment });
  } catch (error) {
    res.status(400).json({ success: false, error: error.message });
  }
});

// Delete assignment
router.delete('/:id', async (req, res) => {
  try {
    const assignment = await Assignment.findByIdAndDelete(req.params.id);
    if (!assignment) {
      return res.status(404).json({ success: false, error: 'Assignment not found' });
    }
    
    // Delete associated files
    assignment.attachments.forEach(attachment => {
      if (fs.existsSync(attachment.path)) {
        fs.unlinkSync(attachment.path);
      }
    });
    
    // Delete all submissions
    await AssignmentSubmission.deleteMany({ assignmentId: assignment._id });
    
    res.json({ success: true, message: 'Assignment deleted successfully' });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
});

// Publish assignment
router.put('/:id/publish', async (req, res) => {
  try {
    const assignment = await Assignment.findById(req.params.id);
    if (!assignment) {
      return res.status(404).json({ success: false, error: 'Assignment not found' });
    }
    
    assignment.status = 'published';
    await assignment.save();
    
    // Emit socket event for real-time notifications
    req.app.get('io').emit('assignmentPublished', {
      assignmentId: assignment._id,
      assignedStudents: assignment.assignedStudents,
      title: assignment.title,
      dueDate: assignment.dueDate
    });
    
    res.json({ success: true, assignment });
  } catch (error) {
    res.status(400).json({ success: false, error: error.message });
  }
});

// Get submissions for an assignment
router.get('/:id/submissions', async (req, res) => {
  try {
    const { status, page = 1, limit = 10 } = req.query;
    const filter = { assignmentId: req.params.id };
    
    if (status) filter.status = status;
    
    const skip = (page - 1) * limit;
    
    const submissions = await AssignmentSubmission.find(filter)
      .sort({ submittedAt: -1 })
      .skip(skip)
      .limit(parseInt(limit));
    
    const total = await AssignmentSubmission.countDocuments(filter);
    
    res.json({
      success: true,
      submissions,
      pagination: {
        page: parseInt(page),
        limit: parseInt(limit),
        total,
        pages: Math.ceil(total / limit)
      }
    });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
});

// Grade submission
router.put('/submissions/:submissionId/grade', async (req, res) => {
  try {
    const { grade, feedback, teacherId } = req.body;
    
    const submission = await AssignmentSubmission.findById(req.params.submissionId);
    if (!submission) {
      return res.status(404).json({ success: false, error: 'Submission not found' });
    }
    
    submission.grade = parseFloat(grade);
    submission.feedback = feedback;
    submission.status = 'graded';
    submission.gradedBy = teacherId;
    submission.gradedAt = new Date();
    
    await submission.save();
    
    // Emit socket event for real-time notifications
    req.app.get('io').emit('assignmentGraded', {
      submissionId: submission._id,
      studentId: submission.studentId,
      grade: submission.grade,
      assignmentId: submission.assignmentId
    });
    
    res.json({ success: true, submission });
  } catch (error) {
    res.status(400).json({ success: false, error: error.message });
  }
});

// Add comment to submission
router.post('/submissions/:submissionId/comments', async (req, res) => {
  try {
    const { comment, isPrivate = false } = req.body;
    
    const submission = await AssignmentSubmission.findById(req.params.submissionId);
    if (!submission) {
      return res.status(404).json({ success: false, error: 'Submission not found' });
    }
    
    await submission.addComment(comment, isPrivate);
    
    res.json({ success: true, submission });
  } catch (error) {
    res.status(400).json({ success: false, error: error.message });
  }
});

// Request revision
router.put('/submissions/:submissionId/request-revision', async (req, res) => {
  try {
    const { deadline, comments } = req.body;
    
    const submission = await AssignmentSubmission.findById(req.params.submissionId);
    if (!submission) {
      return res.status(404).json({ success: false, error: 'Submission not found' });
    }
    
    await submission.requestRevision(new Date(deadline), comments);
    
    // Emit socket event for real-time notifications
    req.app.get('io').emit('revisionRequested', {
      submissionId: submission._id,
      studentId: submission.studentId,
      deadline: deadline,
      assignmentId: submission.assignmentId
    });
    
    res.json({ success: true, submission });
  } catch (error) {
    res.status(400).json({ success: false, error: error.message });
  }
});

// Get assigned students for teacher
router.get('/teacher/:teacherId/students', async (req, res) => {
  try {
    const students = await Student.find({
      'assignedTeachers.teacherId': req.params.teacherId
    }).select('studentName userId assignedTeachers');
    
    const formattedStudents = students.map(student => {
      const teacherAssignment = student.assignedTeachers.find(
        assignment => assignment.teacherId === req.params.teacherId
      );
      
      return {
        studentId: student.userId,
        studentName: student.studentName,
        subject: teacherAssignment ? teacherAssignment.subject : 'N/A'
      };
    });
    
    res.json({ success: true, students: formattedStudents });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
});

// Download assignment attachment
router.get('/:id/attachments/:filename', async (req, res) => {
  try {
    const assignment = await Assignment.findById(req.params.id);
    if (!assignment) {
      return res.status(404).json({ success: false, error: 'Assignment not found' });
    }
    
    const attachment = assignment.attachments.find(att => att.filename === req.params.filename);
    if (!attachment) {
      return res.status(404).json({ success: false, error: 'Attachment not found' });
    }
    
    const filePath = attachment.path;
    if (!fs.existsSync(filePath)) {
      return res.status(404).json({ success: false, error: 'File not found on server' });
    }
    
    res.download(filePath, attachment.originalName);
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
});

// Remove attachment from assignment
router.delete('/:id/attachments/:filename', async (req, res) => {
  try {
    const assignment = await Assignment.findById(req.params.id);
    if (!assignment) {
      return res.status(404).json({ success: false, error: 'Assignment not found' });
    }
    
    const attachmentIndex = assignment.attachments.findIndex(att => att.filename === req.params.filename);
    if (attachmentIndex === -1) {
      return res.status(404).json({ success: false, error: 'Attachment not found' });
    }
    
    const attachment = assignment.attachments[attachmentIndex];
    
    // Delete file from filesystem
    if (fs.existsSync(attachment.path)) {
      fs.unlinkSync(attachment.path);
    }
    
    // Remove from database
    assignment.attachments.splice(attachmentIndex, 1);
    await assignment.save();
    
    res.json({ success: true, message: 'Attachment removed successfully' });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
});

module.exports = router;